import React, { useState, useEffect } from 'react';
import axios from 'axios';
import {
    Container,
    Typography,
    Table,
    TableBody,
    TableCell,
    TableContainer,
    TableHead,
    TableRow,
    Paper,
    IconButton,
    Fab,
    Dialog,
    DialogTitle,
    DialogContent,
    DialogActions,
    Button,
    TextField,
} from '@mui/material';
import { Add, Edit, Delete } from '@mui/icons-material';

function Services() {
    const [services, setServices] = useState([]);
    const [open, setOpen] = useState(false);
    const [currentService, setCurrentService] = useState(null);

    useEffect(() => {
        fetchServices();
    }, []);

    const fetchServices = async () => {
        try {
            const token = localStorage.getItem('token');
            const response = await axios.get('http://localhost:5001/services', {
                headers: { Authorization: `Bearer ${token}` },
            });
            setServices(response.data);
        } catch (error) {
            console.error('Error fetching services:', error.response?.data || error.message);
        }
    };

    const handleClickOpen = (service = null) => {
        setCurrentService(service);
        setOpen(true);
    };

    const handleClose = () => {
        setOpen(false);
        setCurrentService(null);
    };

    const handleInputChange = (e) => {
        const { name, value } = e.target;
        setCurrentService({ ...currentService, [name]: value });
    };

    const handleSaveService = async () => {
        try {
            const token = localStorage.getItem('token');
            if (currentService?.id) {
                await axios.put(`http://localhost:5001/services/${currentService.id}`, currentService, {
                    headers: { Authorization: `Bearer ${token}` },
                });
            } else {
                await axios.post('http://localhost:5001/services', currentService, {
                    headers: { Authorization: `Bearer ${token}` },
                });
            }
            fetchServices();
            handleClose();
        } catch (error) {
            console.error('Error saving service:', error.response?.data || error.message);
        }
    };

    const handleDeleteService = async (id) => {
        try {
            const token = localStorage.getItem('token');
            await axios.delete(`http://localhost:5001/services/${id}`, {
                headers: { Authorization: `Bearer ${token}` },
            });
            fetchServices();
        } catch (error) {
            console.error('Error deleting service:', error.response?.data || error.message);
        }
    };

    return (
        <Container>
            <Typography variant="h4" gutterBottom>
                Services
            </Typography>
            <TableContainer component={Paper}>
                <Table>
                    <TableHead>
                        <TableRow>
                            <TableCell>Name</TableCell>
                            <TableCell>Description</TableCell>
                            <TableCell>Price</TableCell>
                            <TableCell>Duration</TableCell>
                            <TableCell>Actions</TableCell>
                        </TableRow>
                    </TableHead>
                    <TableBody>
                        {services.map((service) => (
                            <TableRow key={service.id}>
                                <TableCell>{service.name}</TableCell>
                                <TableCell>{service.description}</TableCell>
                                <TableCell>${Number(service.price).toFixed(2)}</TableCell>
                                <TableCell>{service.duration}</TableCell>
                                <TableCell>
                                    <IconButton color="primary" onClick={() => handleClickOpen(service)}>
                                        <Edit />
                                    </IconButton>
                                    <IconButton color="secondary" onClick={() => handleDeleteService(service.id)}>
                                        <Delete />
                                    </IconButton>
                                </TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
            </TableContainer>
            <Fab color="primary" aria-label="add" sx={{ position: 'fixed', bottom: 16, right: 16 }} onClick={() => handleClickOpen()}>
                <Add />
            </Fab>
            <Dialog open={open} onClose={handleClose}>
                <DialogTitle>{currentService?.id ? 'Edit Service' : 'Add Service'}</DialogTitle>
                <DialogContent>
                    <TextField
                        label="Name"
                        name="name"
                        fullWidth
                        margin="normal"
                        value={currentService?.name || ''}
                        onChange={handleInputChange}
                    />
                    <TextField
                        label="Description"
                        name="description"
                        fullWidth
                        margin="normal"
                        value={currentService?.description || ''}
                        onChange={handleInputChange}
                    />
                    <TextField
                        label="Price"
                        name="price"
                        fullWidth
                        margin="normal"
                        type="number"
                        value={currentService?.price || ''}
                        onChange={handleInputChange}
                    />
                    <TextField
                        label="Duration"
                        name="duration"
                        fullWidth
                        margin="normal"
                        value={currentService?.duration || ''}
                        onChange={handleInputChange}
                    />
                </DialogContent>
                <DialogActions>
                    <Button onClick={handleClose} color="secondary">
                        Cancel
                    </Button>
                    <Button onClick={handleSaveService} color="primary">
                        Save
                    </Button>
                </DialogActions>
            </Dialog>
        </Container>
    );
}

export default Services;
