const express = require('express');
const cors = require('cors');
const mysql = require('mysql2/promise');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const bodyParser = require('body-parser');
const { body, validationResult } = require('express-validator'); 
const querystring = require('querystring');
const session = require('express-session');
const axios = require('axios');
require('dotenv').config();

console.log(process.env.FRONTEND_URL);
const app = express();
app.use(bodyParser.json());
app.use(cors({
    origin: process.env.FRONTEND_URL || 'http://localhost:3000',
    credentials: true,
}));
app.use(express.json());

// Configure session middleware
app.use(session({
    secret: 'c35ec2ad9e066f6439ed87ac429de5c381b9a63a47bc5c1b3620f7de835e61f0aff870c2da9a99e460d60b6524e14b1b7d85f403fe9dd87ad9fa9e4202e38c49', // Replace with a strong secret
    resave: false,
    saveUninitialized: true,
    cookie: { secure: false } // Set to true if using HTTPS
}));

// JWT Secret Key
const SECRET_KEY = process.env.SECRET_KEY || 'your_jwt_secret_key';
 
// Load environment variables
const dbConfig = {
    host: process.env.DB_HOST,
    user: process.env.DB_USER,
    password: process.env.DB_PASSWORD,
    database: process.env.DB_NAME,
    port: process.env.DB_PORT || 3306,
};
console.log('Loaded environment variables:', dbConfig);

// Create the MySQL connection
const db = mysql.createPool(dbConfig);

// Connect to the database
(async () => {
    try {
        const connection = await db.getConnection(); // Get a connection from the pool
        console.log('Connected to MySQL database');
        connection.release(); // Release the connection back to the pool
    } catch (err) {
        console.error('Database connection failed:', err.message);
        process.exit(1); // Exit if the database connection fails
    }
})();

// Session Middleware
app.use(
    session({
        secret: process.env.SESSION_SECRET || 'your_session_secret',
        resave: false,
        saveUninitialized: true,
        cookie: { secure: false }, // Set to `true` if using HTTPS
    })
);

// JWT Middleware
const authenticateToken = (req, res, next) => {
    const authHeader = req.headers['authorization'];
    const token = authHeader && authHeader.split(' ')[1];

    if (!token) {
        return res.status(401).json({ error: 'Access token required' });
    }

    jwt.verify(token, SECRET_KEY, (err, user) => {
        if (err) {
            if (err.name === 'TokenExpiredError') {
                console.error('Token expired:', err);
                return res.status(401).json({ error: 'Token expired' });
            }
            console.error('Invalid token:', err);
            return res.status(403).json({ error: 'Invalid token' });
        }
        req.user = user; // Attach user info to request object
        next();
    });
};
// User Registration
app.post(
    '/register',
    [
        body('username').notEmpty().withMessage('Username is required'),
        body('email').isEmail().withMessage('Valid email is required'),
        body('password').isLength({ min: 6 }).withMessage('Password must be at least 6 characters'),
    ],
    async (req, res) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({ errors: errors.array() });
        }

        const { username, password, email } = req.body;
        const hashedPassword = bcrypt.hashSync(password, 10);

        const query = 'INSERT INTO users (username, password, email, isAdmin) VALUES (?, ?, ?, ?)';

        try {
            const connection = await db.getConnection(); // Adjust to your `mysql2/promise` connection method
            await connection.execute(query, [username, hashedPassword, email, false]);
            await connection.release(); // Release the connection
            res.status(201).json({ message: 'User registered successfully' });
        } catch (err) {
            if (err.code === 'ER_DUP_ENTRY') {
                return res.status(400).json({ error: 'Username or email already exists' });
            }
            res.status(500).json({ error: err.message });
        }
    }
);

app.get('/test', (req, res) => {
    res.send('Backend is working!');
});

// Create a route that sends a response when visiting the homepage
app.get('/', (req, res) => {
    console.log('CHECKING>>>');
     res.send('<h1>Hello, BACKEND Server running!</h1>'); 
   });
   
   // User Registration
app.post(
    '/register',
    [
        body('username').notEmpty().withMessage('Username is required'),
        body('email').isEmail().withMessage('Valid email is required'),
        body('password').isLength({ min: 6 }).withMessage('Password must be at least 6 characters'),
    ],
    async (req, res) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({ errors: errors.array() });
        }

        const { username, password, email } = req.body;
        const hashedPassword = bcrypt.hashSync(password, 10);

        const query = 'INSERT INTO users (username, password, email, isAdmin) VALUES (?, ?, ?, ?)';

        try {
            const connection = await db.getConnection(); // Adjust to your `mysql2/promise` connection method
            await connection.execute(query, [username, hashedPassword, email, false]);
            await connection.release(); // Release the connection
            res.status(201).json({ message: 'User registered successfully' });
        } catch (err) {
            if (err.code === 'ER_DUP_ENTRY') {
                return res.status(400).json({ error: 'Username or email already exists' });
            }
            res.status(500).json({ error: err.message });
        }
    }
);


// User Login
app.post('/login', async (req, res) => {
    console.log('Login request body:', req.body);
    const { username, password } = req.body;
    try {
        const connection = await db.getConnection(); // Adjust this to your connection setup
        const [results] = await connection.execute('SELECT * FROM users WHERE username = ?', [username]);
        console.log('Query results:', results);

        if (results.length === 0) {
            console.log('User not found');
            await connection.release(); // Release connection before returning
            return res.status(404).json({ error: 'User not found' });
        }

        const user = results[0];
        const isValid = bcrypt.compareSync(password, user.password);
        console.log('Password validation:', isValid);

        if (isValid) {
            const payload = { id: user.id, username: user.username, isAdmin: user.isAdmin };
            const accessToken = jwt.sign(payload, process.env.SECRET_KEY, { expiresIn: '1h' });
            console.log('Generated accessToken:', accessToken);

            await connection.release(); // Release connection before returning
            res.json({ message: 'Login successful', accessToken });
        } else {
            console.log('Invalid password');
            await connection.release(); // Release connection before returning
            res.status(401).json({ error: 'Invalid password' });
        }
    } catch (err) {
        console.error('Error during login:', err.message);
        res.status(500).json({ error: 'Internal server error' });
    }
});


// Clients API
app.use('/clients', authenticateToken);

// Get all clients
app.get('/clients', async (req, res) => {
    const userId = req.user.id;
    const query = 'SELECT * FROM clients WHERE user_id = ?';

    try {
        const [results] = await db.execute(query, [userId]);
        res.json(results);
    } catch (err) {
        console.error('Error fetching clients:', err);
        res.status(500).json({ error: err.message });
    }
});

// Add a new client
app.post(
    '/clients',
    [
        body('name').notEmpty().withMessage('Name is required'),
        body('address').notEmpty().withMessage('Address is required'),
        body('contact').notEmpty().withMessage('Contact is required'),
    ],
    async (req, res) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({ errors: errors.array() });
        }

        const { name, address, contact } = req.body;
        const userId = req.user.id;

        const query = 'INSERT INTO clients (name, address, contact, user_id) VALUES (?, ?, ?, ?)';
        try {
            const [result] = await db.execute(query, [name, address, contact, userId]);
            res.status(201).json({ message: 'Client added successfully', clientId: result.insertId });
        } catch (err) {
            console.error('Error adding client:', err);
            res.status(500).json({ error: err.message });
        }
    }
);

// Delete a client
app.delete('/clients/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    try {
        // First, delete related estimates
        const deleteEstimatesQuery = 'DELETE FROM estimates WHERE client_id = ?';
        await db.execute(deleteEstimatesQuery, [id]);

        // Then, delete the client
        const deleteClientQuery = 'DELETE FROM clients WHERE id = ? AND user_id = ?';
        const [result] = await db.execute(deleteClientQuery, [id, userId]);

        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Client not found or unauthorized' });
        }
        res.status(204).send();
    } catch (err) {
        console.error('Error deleting client:', err);
        res.status(500).json({ error: 'Error deleting client' });
    }
});

// Update a client
app.put(
    '/clients/:id',
    authenticateToken,
    [
        body('name').notEmpty().withMessage('Name is required'),
        body('address').notEmpty().withMessage('Address is required'),
        body('contact').notEmpty().withMessage('Contact is required'),
    ],
    async (req, res) => {
        const { id } = req.params; // Client ID
        const userId = req.user.id; // Authenticated User ID
        const errors = validationResult(req);

        // Validate input
        if (!errors.isEmpty()) {
            return res.status(400).json({ errors: errors.array() });
        }

        const { name, address, contact } = req.body;

        // SQL Query to update the client
        const query = `UPDATE clients SET name = ?, address = ?, contact = ?, updated_at = NOW() 
                       WHERE id = ? AND user_id = ?`;

        try {
            const [result] = await db.execute(query, [name, address, contact, id, userId]);

            if (result.affectedRows === 0) {
                return res.status(404).json({ error: 'Client not found or unauthorized' });
            }
            res.json({ message: 'Client updated successfully' });
        } catch (err) {
            console.error('Error updating client:', err);
            res.status(500).json({ error: 'Error updating client' });
        }
    }
);

// Refresh token endpoint
app.post('/refresh-token', (req, res) => {
    const { refreshToken } = req.body;

    if (!refreshToken) {
        return res.status(401).json({ error: 'Refresh token required' });
    }

    // Verify the refresh token
    jwt.verify(refreshToken, SECRET_KEY, (err, user) => {
        if (err) {
            return res.status(403).json({ error: 'Invalid refresh token' });
        }

        // Generate a new access token
        const { id, username, isAdmin } = user;
        const newAccessToken = jwt.sign({ id, username, isAdmin }, SECRET_KEY, { expiresIn: '1h' });

        res.json({ accessToken: newAccessToken });
    });
});

// Start Server
const PORT = process.env.PORT || 5001;
const hostname = process.env.FRONTEND_URL || 'https://new.landscapingsoftwarepro.com';
app.listen(PORT, () => {
    console.log(`Server running at ${hostname}:${PORT}/`);
    
    //console.log(`Server running on port ${PORT}`);
});

// Get all jobs
app.get('/jobs', authenticateToken, async (req, res) => {
    const userId = req.user.id;
    const query = `
        SELECT jobs.*, clients.name AS client_name 
        FROM jobs 
        INNER JOIN clients ON jobs.client_id = clients.id 
        WHERE jobs.user_id = ?`;

    try {
        const [results] = await db.execute(query, [userId]);
        res.json(results);
    } catch (err) {
        console.error('Error fetching jobs:', err);
        res.status(500).json({ error: err.message });
    }
});

// Create a job
app.post('/jobs', authenticateToken, async (req, res) => {
    const { client_id, name, description, status, start_date, end_date } = req.body;
    const userId = req.user.id;

    const query = `
        INSERT INTO jobs (user_id, client_id, name, description, status, start_date, end_date)
        VALUES (?, ?, ?, ?, ?, ?, ?)`;

    try {
        const [result] = await db.execute(query, [
            userId, client_id, name, description, status, start_date, end_date,
        ]);
        res.status(201).json({ message: 'Job created successfully', jobId: result.insertId });
    } catch (err) {
        console.error('Error creating job:', err);
        res.status(500).json({ error: err.message });
    }
});

// Get a specific job
app.get('/jobs/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = `
        SELECT jobs.*, clients.name AS client_name 
        FROM jobs 
        INNER JOIN clients ON jobs.client_id = clients.id 
        WHERE jobs.id = ? AND jobs.user_id = ?`;

    try {
        const [results] = await db.execute(query, [id, userId]);
        if (results.length === 0) {
            return res.status(404).json({ error: 'Job not found or unauthorized' });
        }
        res.json(results[0]);
    } catch (err) {
        console.error('Error fetching job details:', err);
        res.status(500).json({ error: err.message });
    }
});

// Update a job
app.put('/jobs/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const { client_id, name, description, status, start_date, end_date } = req.body;
    const userId = req.user.id;

    const query = `
        UPDATE jobs 
        SET client_id = ?, name = ?, description = ?, status = ?, start_date = ?, end_date = ? 
        WHERE id = ? AND user_id = ?`;

    try {
        const [result] = await db.execute(query, [
            client_id, name, description, status, start_date, end_date, id, userId,
        ]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Job not found or unauthorized' });
        }
        res.json({ message: 'Job updated successfully' });
    } catch (err) {
        console.error('Error updating job:', err);
        res.status(500).json({ error: err.message });
    }
});

// Delete a job
app.delete('/jobs/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = 'DELETE FROM jobs WHERE id = ? AND user_id = ?';

    try {
        const [result] = await db.execute(query, [id, userId]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Job not found or unauthorized' });
        }
        res.status(204).send();
    } catch (err) {
        console.error('Error deleting job:', err);
        res.status(500).json({ error: err.message });
    }
});

// LEADS API

// Get all leads for the authenticated user
app.get('/leads', authenticateToken, async (req, res) => {
    const userId = req.user.id;
    const query = 'SELECT * FROM leads WHERE user_id = ?';

    try {
        const [results] = await db.execute(query, [userId]);
        res.json(results);
    } catch (err) {
        console.error('Error fetching leads:', err);
        res.status(500).json({ error: err.message });
    }
});

// Add a new lead
app.post('/leads', authenticateToken, async (req, res) => {
    const { name, phone, email, address } = req.body;
    const userId = req.user.id;
    const query = 'INSERT INTO leads (name, phone, email, address, user_id) VALUES (?, ?, ?, ?, ?)';

    try {
        const [result] = await db.execute(query, [name, phone, email, address, userId]);
        res.status(201).json({ message: 'Lead added successfully', leadId: result.insertId });
    } catch (err) {
        console.error('Error adding lead:', err);
        res.status(500).json({ error: err.message });
    }
});

// Update a lead
app.put('/leads/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const { name, phone, email, address } = req.body;
    const userId = req.user.id;
    const query = 'UPDATE leads SET name = ?, phone = ?, email = ?, address = ? WHERE id = ? AND user_id = ?';

    try {
        const [result] = await db.execute(query, [name, phone, email, address, id, userId]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Lead not found or unauthorized' });
        }
        res.json({ message: 'Lead updated successfully' });
    } catch (err) {
        console.error('Error updating lead:', err);
        res.status(500).json({ error: err.message });
    }
});

// Delete a lead
app.delete('/leads/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;
    const query = 'DELETE FROM leads WHERE id = ? AND user_id = ?';

    try {
        const [result] = await db.execute(query, [id, userId]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Lead not found or unauthorized' });
        }
        res.status(204).send();
    } catch (err) {
        console.error('Error deleting lead:', err);
        res.status(500).json({ error: err.message });
    }
});

// Fetch all services for the logged-in user
app.get('/services', authenticateToken, async (req, res) => {
    const userId = req.user.id;
    const query = 'SELECT * FROM services WHERE user_id = ?';

    try {
        const [results] = await db.execute(query, [userId]);
        res.json(results);
    } catch (err) {
        console.error('Error fetching services:', err);
        res.status(500).json({ error: err.message });
    }
});

// Add a new service
app.post('/services', authenticateToken, async (req, res) => {
    const { name, description, price, duration } = req.body;
    const userId = req.user.id;

    if (!name || !description || !price || !duration) {
        return res.status(400).json({ error: 'All fields are required' });
    }

    const query = 'INSERT INTO services (name, description, price, duration, user_id) VALUES (?, ?, ?, ?, ?)';

    try {
        const [result] = await db.execute(query, [name, description, price, duration, userId]);
        res.status(201).json({ message: 'Service added successfully', serviceId: result.insertId });
    } catch (err) {
        console.error('Error adding service:', err);
        res.status(500).json({ error: err.message });
    }
});

// Update an existing service
app.put('/services/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const { name, description, price, duration } = req.body;
    const userId = req.user.id;

    const query = 'UPDATE services SET name = ?, description = ?, price = ?, duration = ? WHERE id = ? AND user_id = ?';

    try {
        const [result] = await db.execute(query, [name, description, price, duration, id, userId]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Service not found or unauthorized' });
        }
        res.json({ message: 'Service updated successfully' });
    } catch (err) {
        console.error('Error updating service:', err);
        res.status(500).json({ error: err.message });
    }
});

// Delete a service
app.delete('/services/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = 'DELETE FROM services WHERE id = ? AND user_id = ?';

    try {
        const [result] = await db.execute(query, [id, userId]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Service not found or unauthorized' });
        }
        res.status(204).send();
    } catch (err) {
        console.error('Error deleting service:', err);
        res.status(500).json({ error: err.message });
    }
});

// Invoices API

// Fetch all invoices for the logged-in user
app.get('/invoices', authenticateToken, async (req, res) => {
    const userId = req.user.id;
    const query = `
        SELECT invoices.*, clients.name AS client_name
        FROM invoices
        INNER JOIN clients ON invoices.client_id = clients.id
        WHERE invoices.user_id = ?
        ORDER BY invoices.issue_date DESC;
    `;

    try {
        const [results] = await db.execute(query, [userId]);
        res.json(results);
    } catch (err) {
        console.error('Error fetching invoices:', err);
        res.status(500).json({ error: 'Failed to fetch invoices' });
    }
});

// Add a new invoice
app.post('/invoices', authenticateToken, async (req, res) => {
    const { client_id, invoice_number, issue_date, due_date, total_amount } = req.body;
    const userId = req.user.id;

    const query = `
        INSERT INTO invoices (user_id, client_id, invoice_number, issue_date, due_date, total_amount)
        VALUES (?, ?, ?, ?, ?, ?);
    `;

    try {
        const [result] = await db.execute(query, [userId, client_id, invoice_number, issue_date, due_date, total_amount]);
        res.status(201).json({ message: 'Invoice added successfully', invoiceId: result.insertId });
    } catch (err) {
        console.error('Error adding invoice:', err);
        res.status(500).json({ error: 'Failed to add invoice' });
    }
});

// Update an existing invoice
app.put('/invoices/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const { client_id, invoice_number, issue_date, due_date, total_amount, status } = req.body;
    const userId = req.user.id;

    const query = `
        UPDATE invoices
        SET client_id = ?, invoice_number = ?, issue_date = ?, due_date = ?, total_amount = ?, status = ?
        WHERE id = ? AND user_id = ?;
    `;

    try {
        const [result] = await db.execute(query, [
            client_id,
            invoice_number,
            issue_date,
            due_date,
            total_amount,
            status,
            id,
            userId,
        ]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Invoice not found or unauthorized' });
        }
        res.json({ message: 'Invoice updated successfully' });
    } catch (err) {
        console.error('Error updating invoice:', err);
        res.status(500).json({ error: 'Failed to update invoice' });
    }
});

// Delete an invoice
app.delete('/invoices/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = `
        DELETE FROM invoices
        WHERE id = ? AND user_id = ?;
    `;

    try {
        const [result] = await db.execute(query, [id, userId]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Invoice not found or unauthorized' });
        }
        res.json({ message: 'Invoice deleted successfully' });
    } catch (err) {
        console.error('Error deleting invoice:', err);
        res.status(500).json({ error: 'Failed to delete invoice' });
    }
});

// Mark an invoice as paid
app.put('/invoices/:id/mark-paid', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = `UPDATE invoices SET status = 'Paid' WHERE id = ? AND user_id = ?`;

    try {
        const [result] = await db.execute(query, [id, userId]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Invoice not found or unauthorized' });
        }
        res.json({ message: 'Invoice marked as paid successfully' });
    } catch (err) {
        console.error('Error updating invoice status:', err);
        res.status(500).json({ error: 'Error updating invoice status' });
    }
});

// Fetch details of a specific invoice
app.get('/invoices/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = `
        SELECT invoices.*, clients.name AS client_name, clients.contact AS client_contact
        FROM invoices
        INNER JOIN clients ON invoices.client_id = clients.id
        WHERE invoices.id = ? AND invoices.user_id = ?;
    `;

    try {
        const [results] = await db.execute(query, [id, userId]);
        if (results.length === 0) {
            return res.status(404).json({ error: 'Invoice not found' });
        }
        res.json(results[0]);
    } catch (err) {
        console.error('Error fetching invoice details:', err);
        res.status(500).json({ error: 'Failed to fetch invoice details' });
    }
});

// Get all expenses for a user
app.get('/expenses', authenticateToken, async (req, res) => {
    const userId = req.user.id;
    const query = 'SELECT * FROM expenses WHERE user_id = ? ORDER BY date DESC';

    try {
        const [results] = await db.execute(query, [userId]);
        res.json(results);
    } catch (err) {
        console.error('Error fetching expenses:', err);
        res.status(500).json({ error: err.message });
    }
});

// Add a new expense
app.post('/expenses', authenticateToken, async (req, res) => {
    const { name, category, date, amount, description } = req.body;
    const userId = req.user.id;

    const query = 'INSERT INTO expenses (user_id, name, category, date, amount, description) VALUES (?, ?, ?, ?, ?, ?)';

    try {
        const [result] = await db.execute(query, [userId, name, category, date, amount, description]);
        res.status(201).json({ message: 'Expense added successfully', expenseId: result.insertId });
    } catch (err) {
        console.error('Error adding expense:', err);
        res.status(500).json({ error: err.message });
    }
});

// Update an expense
app.put('/expenses/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const { name, category, date, amount, description } = req.body;
    const userId = req.user.id;

    const query = 'UPDATE expenses SET name = ?, category = ?, date = ?, amount = ?, description = ? WHERE id = ? AND user_id = ?';

    try {
        const [result] = await db.execute(query, [name, category, date, amount, description, id, userId]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Expense not found or unauthorized' });
        }
        res.json({ message: 'Expense updated successfully' });
    } catch (err) {
        console.error('Error updating expense:', err);
        res.status(500).json({ error: err.message });
    }
});

// Delete an expense
app.delete('/expenses/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = 'DELETE FROM expenses WHERE id = ? AND user_id = ?';

    try {
        const [result] = await db.execute(query, [id, userId]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Expense not found or unauthorized' });
        }
        res.status(204).send();
    } catch (err) {
        console.error('Error deleting expense:', err);
        res.status(500).json({ error: err.message });
    }
});

// Get all estimates for the logged-in user
app.get('/estimates', authenticateToken, async (req, res) => {
    const userId = req.user.id;

    const query = `
        SELECT 
            estimates.*, 
            clients.name AS client_name, 
            JSON_ARRAYAGG(
                JSON_OBJECT(
                    'id', estimate_items.id, 
                    'description', estimate_items.description, 
                    'quantity', estimate_items.quantity, 
                    'unit_price', estimate_items.unit_price
                )
            ) AS items
        FROM estimates
        INNER JOIN clients ON estimates.client_id = clients.id
        LEFT JOIN estimate_items ON estimates.id = estimate_items.estimate_id
        WHERE estimates.user_id = ?
        GROUP BY estimates.id;
    `;

    try {
        const [results] = await db.execute(query, [userId]);
        res.json(results);
    } catch (err) {
        console.error('Error fetching estimates:', err);
        res.status(500).json({ error: 'Error fetching estimates' });
    }
});

// Add a new estimate
app.post('/estimates', authenticateToken, async (req, res) => {
    const userId = req.user.id;
    const { client_id, date, total_amount, items } = req.body;

    if (!client_id || !date || !total_amount || !items || items.length === 0) {
        return res.status(400).json({ error: 'All fields, including items, are required.' });
    }

    const query = `
        INSERT INTO estimates (client_id, user_id, date, total_amount)
        VALUES (?, ?, ?, ?)
    `;

    try {
        const [result] = await db.execute(query, [client_id, userId, date, total_amount]);
        const estimateId = result.insertId;

        // Insert items into `estimate_items`
        const itemQueries = items.map((item) => {
            const itemQuery = `
                INSERT INTO estimate_items (estimate_id, description, quantity, unit_price)
                VALUES (?, ?, ?, ?)
            `;
            return db.execute(itemQuery, [estimateId, item.description, item.quantity, item.unit_price]);
        });

        await Promise.all(itemQueries);
        res.status(201).json({ message: 'Estimate added successfully', estimateId });
    } catch (err) {
        console.error('Error adding estimate:', err);
        res.status(500).json({ error: 'Error adding estimate' });
    }
});

// Update an estimate
app.put('/estimates/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const { client_id, date, total_amount, items } = req.body;
    const userId = req.user.id;

    if (!client_id || !date || !total_amount || !items || items.length === 0) {
        return res.status(400).json({ error: 'All fields, including items, are required.' });
    }

    const query = `
        UPDATE estimates
        SET client_id = ?, date = ?, total_amount = ?
        WHERE id = ? AND user_id = ?
    `;

    try {
        const [result] = await db.execute(query, [client_id, date, total_amount, id, userId]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Estimate not found or unauthorized' });
        }

        // Delete existing items and insert new ones
        await db.execute(`DELETE FROM estimate_items WHERE estimate_id = ?`, [id]);

        const itemQueries = items.map((item) => {
            const itemQuery = `
                INSERT INTO estimate_items (estimate_id, description, quantity, unit_price)
                VALUES (?, ?, ?, ?)
            `;
            return db.execute(itemQuery, [id, item.description, item.quantity, item.unit_price]);
        });

        await Promise.all(itemQueries);
        res.json({ message: 'Estimate updated successfully' });
    } catch (err) {
        console.error('Error updating estimate:', err);
        res.status(500).json({ error: 'Error updating estimate' });
    }
});

// Delete an estimate
app.delete('/estimates/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    try {
        await db.execute('DELETE FROM estimate_items WHERE estimate_id = ?', [id]);

        const [result] = await db.execute('DELETE FROM estimates WHERE id = ? AND user_id = ?', [id, userId]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Estimate not found or unauthorized' });
        }

        res.json({ message: 'Estimate deleted successfully' });
    } catch (err) {
        console.error('Error deleting estimate:', err);
        res.status(500).json({ error: 'Error deleting estimate' });
    }
});

// Fetch details of a specific estimate
app.get('/estimates/:id', authenticateToken, async (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = `
        SELECT estimates.*, clients.name AS client_name
        FROM estimates
        INNER JOIN clients ON estimates.client_id = clients.id
        WHERE estimates.id = ? AND estimates.user_id = ?
    `;

    try {
        const [estimateResults] = await db.execute(query, [id, userId]);
        if (estimateResults.length === 0) {
            return res.status(404).json({ error: 'Estimate not found or unauthorized' });
        }

        const estimate = estimateResults[0];

        const itemsQuery = `SELECT * FROM estimate_items WHERE estimate_id = ?`;
        const [items] = await db.execute(itemsQuery, [id]);

        res.json({ ...estimate, items });
    } catch (err) {
        console.error('Error fetching estimate details:', err);
        res.status(500).json({ error: 'Error fetching estimate details' });
    }
});

// Reports API

// Fetch total income for the logged-in user
app.get('/reports/total-income', authenticateToken, async (req, res) => {
    const userId = req.user.id;

    const query = `SELECT SUM(total_amount) AS total_income FROM invoices WHERE user_id = ?`;

    try {
        const [results] = await db.execute(query, [userId]);
        res.json({ total_income: results[0]?.total_income || 0 });
    } catch (err) {
        console.error('Error fetching total income:', err);
        res.status(500).json({ error: 'Error fetching total income' });
    }
});

// Fetch total expenses for the logged-in user
app.get('/reports/total-expenses', authenticateToken, async (req, res) => {
    const userId = req.user.id;

    const query = `SELECT SUM(amount) AS total_expenses FROM expenses WHERE user_id = ?`;

    try {
        const [results] = await db.execute(query, [userId]);
        res.json({ total_expenses: results[0]?.total_expenses || 0 });
    } catch (err) {
        console.error('Error fetching total expenses:', err);
        res.status(500).json({ error: 'Error fetching total expenses' });
    }
});

// Reports Chart Data API
app.get('/reports/chart-data', authenticateToken, async (req, res) => {
    const userId = req.user.id;
    const filter = req.query.filter || 'monthly';

    let incomeQuery = '';
    let expensesQuery = '';

    // Determine the grouping based on the filter
    if (filter === 'weekly') {
        incomeQuery = `
            SELECT YEARWEEK(issue_date) AS label, SUM(total_amount) AS total_income 
            FROM invoices 
            WHERE user_id = ? AND status = 'Paid'
            GROUP BY YEARWEEK(issue_date)
            ORDER BY label
        `;
        expensesQuery = `
            SELECT YEARWEEK(date) AS label, SUM(amount) AS total_expenses 
            FROM expenses 
            WHERE user_id = ?
            GROUP BY YEARWEEK(date)
            ORDER BY label
        `;
    } else if (filter === 'yearly') {
        incomeQuery = `
            SELECT YEAR(issue_date) AS label, SUM(total_amount) AS total_income 
            FROM invoices 
            WHERE user_id = ? AND status = 'Paid'
            GROUP BY YEAR(issue_date)
            ORDER BY label
        `;
        expensesQuery = `
            SELECT YEAR(date) AS label, SUM(amount) AS total_expenses 
            FROM expenses 
            WHERE user_id = ?
            GROUP BY YEAR(date)
            ORDER BY label
        `;
    } else { // Default to monthly
        incomeQuery = `
            SELECT DATE_FORMAT(issue_date, "%Y-%m") AS label, SUM(total_amount) AS total_income 
            FROM invoices 
            WHERE user_id = ? AND status = 'Paid'
            GROUP BY DATE_FORMAT(issue_date, "%Y-%m")
            ORDER BY label
        `;
        expensesQuery = `
            SELECT DATE_FORMAT(date, "%Y-%m") AS label, SUM(amount) AS total_expenses 
            FROM expenses 
            WHERE user_id = ?
            GROUP BY DATE_FORMAT(date, "%Y-%m")
            ORDER BY label
        `;
    }

    try {
        const [incomeResults] = await db.execute(incomeQuery, [userId]);
        const [expenseResults] = await db.execute(expensesQuery, [userId]);

        // Prepare the combined data
        const labels = [];
        const incomeData = [];
        const expenseData = [];

        // Create a map for faster matching
        const incomeMap = new Map(incomeResults.map(row => [row.label, row.total_income]));
        const expenseMap = new Map(expenseResults.map(row => [row.label, row.total_expenses]));

        const allLabels = Array.from(new Set([...incomeMap.keys(), ...expenseMap.keys()])).sort();

        allLabels.forEach(label => {
            labels.push(label);
            incomeData.push(incomeMap.get(label) || 0);
            expenseData.push(expenseMap.get(label) || 0);
        });

        // Return the data in the correct format
        res.json({
            labels,
            income: incomeData,
            expenses: expenseData,
        });
    } catch (err) {
        console.error('Error fetching chart data:', err);
        res.status(500).json({ error: 'Error fetching chart data' });
    }
});

// QuickBooks Configuration
const quickbooksAuthUrl = "https://appcenter.intuit.com/connect/oauth2";

const quickbooksApiBaseUrl = process.env.QUICKBOOKS_ENV === 'sandbox'
    ? "https://sandbox-quickbooks.api.intuit.com/v3/company"
    : "https://quickbooks.api.intuit.com/v3/company";

const quickbooksTokenUrl = "https://oauth.platform.intuit.com/oauth2/v1/tokens/bearer";

// Generate a random state for CSRF protection
const generateState = () => Math.random().toString(36).substring(2);

// Refresh QuickBooks Token
const refreshQuickBooksToken = async (userId) => {
    const [[tokens]] = await db.execute(
        'SELECT refresh_token FROM quickbooks_tokens WHERE user_id = ?',
        [userId]
    );

    if (!tokens || !tokens.refresh_token) {
        throw new Error('Refresh token not found.');
    }

    const { refresh_token } = tokens;

    const response = await axios.post(
        quickbooksTokenUrl,
        querystring.stringify({
            grant_type: 'refresh_token',
            refresh_token,
            client_id: process.env.QUICKBOOKS_CLIENT_ID,
            client_secret: process.env.QUICKBOOKS_CLIENT_SECRET,
        }),
        {
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        }
    );

    const { access_token, refresh_token: new_refresh_token } = response.data;

    // Update tokens in the database
    await db.execute(
        'UPDATE quickbooks_tokens SET access_token = ?, refresh_token = ?, updated_at = NOW() WHERE user_id = ?',
        [access_token, new_refresh_token, userId]
    );

    return access_token;
};

// Function to refresh token if needed
const refreshAccessTokenIfNeeded = async (userId) => {
    const [[tokens]] = await db.execute(
        'SELECT access_token, refresh_token, updated_at FROM quickbooks_tokens WHERE user_id = ?',
        [userId]
    );

    if (!tokens || !tokens.access_token || !tokens.refresh_token) {
        throw new Error('Tokens not found for user.');
    }

    // Check if the token has expired
    const tokenLifetime = 3600; // QuickBooks token lifetime in seconds
    const lastUpdated = new Date(tokens.updated_at).getTime();
    const now = Date.now();

    const isTokenExpired = (now - lastUpdated) / 1000 > tokenLifetime;

    if (isTokenExpired) {
        console.log('Access token expired. Refreshing token...');
        return await refreshQuickBooksToken(userId);
    }

    return tokens.access_token;
};

// Route to initiate QuickBooks connection
app.get('/quickbooks/callback', async (req, res) => {
    try {
        console.log('Received callback query:', req.query); // Log incoming parameters

        const { code, state, realmId } = req.query;
        if (!code || !state || !realmId) {
            console.error('Missing query parameters');
            return res.status(400).send('Missing required query parameters.');
        }

        // Exchange authorization code for tokens
        const response = await axios.post(
            quickbooksTokenUrl,
            querystring.stringify({
                grant_type: 'authorization_code',
                code,
                redirect_uri: process.env.QUICKBOOKS_REDIRECT_URI,
                client_id: process.env.QUICKBOOKS_CLIENT_ID,
                client_secret: process.env.QUICKBOOKS_CLIENT_SECRET,
            }),
            {
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            }
        );

        const { access_token, refresh_token } = response.data;
        console.log('Tokens received:', { access_token, refresh_token, realmId });

        // Save tokens in the database
        const userId = req.user?.id || 1; // Use a valid user ID dynamically or fallback for testing
        await db.execute(
            `INSERT INTO quickbooks_tokens (user_id, access_token, refresh_token, realm_id) 
             VALUES (?, ?, ?, ?) 
             ON DUPLICATE KEY UPDATE access_token = ?, refresh_token = ?, updated_at = NOW()`,
            [userId, access_token, refresh_token, realmId, access_token, refresh_token]
        );
        console.log('Tokens saved to database.');

        // Redirect to dashboard or success page
        res.redirect('https://old.landscapingsoftwarepro.com/user/expenses');
    } catch (error) {
        console.error('Error during QuickBooks callback processing:', error.message);
        if (error.response) {
            console.error('Error response data:', error.response.data);
        }
        res.status(500).send('Failed to connect with QuickBooks.');
    }
});

app.get('/quickbooks/connect', (req, res) => {
    const state = Math.random().toString(36).substring(2); // Generate a random state for CSRF protection
    req.session.quickbooksState = state;

    const queryParams = querystring.stringify({
        client_id: process.env.QUICKBOOKS_CLIENT_ID,
        redirect_uri: process.env.QUICKBOOKS_REDIRECT_URI,
        response_type: 'code',
        scope: 'com.intuit.quickbooks.accounting',
        state,
    });

    res.redirect(`${quickbooksAuthUrl}?${queryParams}`);
});


// Route to fetch QuickBooks income report
app.get('/quickbooks/income', authenticateToken, async (req, res) => {
    try {
        const [[tokens]] = await db.execute(
            'SELECT access_token, realm_id FROM quickbooks_tokens WHERE user_id = ?',
            [req.user.id]
        );

        if (!tokens) {
            throw new Error('QuickBooks not connected.');
        }

        const { access_token, realm_id: realmId } = tokens;

        const response = await axios.get(
            `${quickbooksApiBaseUrl}/${realmId}/reports/ProfitAndLoss`,
            {
                headers: {
                    Authorization: `Bearer ${access_token}`,
                    Accept: 'application/json',
                },
            }
        );

        res.json(response.data);
    } catch (error) {
        console.error('Error fetching income data:', error.message);
        res.status(500).send('Failed to fetch income data.');
    }
});

//Quickbooks Expenses
app.get('/quickbooks/expenses', authenticateToken, async (req, res) => {
    try {
        const access_token = await refreshAccessTokenIfNeeded(req.user.id);
        const [[tokens]] = await db.execute(
            'SELECT realm_id FROM quickbooks_tokens WHERE user_id = ?',
            [req.user.id]
        );

        if (!tokens || !tokens.realm_id) {
            throw new Error('QuickBooks realm ID not found.');
        }

        const { realm_id: realmId } = tokens;

        console.log('Fetching QuickBooks expenses for realmId:', realmId);

        const response = await axios.get(
            `${quickbooksApiBaseUrl}/${realmId}/query?query=SELECT * FROM Purchase`,
            {
                headers: {
                    Authorization: `Bearer ${access_token}`,
                    Accept: 'application/json',
                },
            }
        );

        console.log('QuickBooks response data:', response.data);

        const quickBooksExpenses = (response.data.QueryResponse.Purchase || []).map(expense => ({
            name: expense.PrivateNote || 'No description',
            category: 'QuickBooks',
            date: expense.TxnDate,
            amount: expense.TotalAmt,
            description: expense.PrivateNote || 'No description',
            source: 'QuickBooks',
        }));

        // Fetch manual expenses
        const [manualExpenses] = await db.execute('SELECT * FROM expenses WHERE user_id = ?', [req.user.id]);
        console.log('Manual Expenses:', manualExpenses);

        // Combine manual and QuickBooks expenses
        const allExpenses = [...quickBooksExpenses, ...manualExpenses];
        res.json(allExpenses); // Send both combined expenses to the frontend
    } catch (error) {
        console.error('Error fetching QuickBooks expenses:', error.message);

        // Handle invalid grant (refresh token expired or invalid)
        if (error.response && error.response.data && error.response.data.error === 'invalid_grant') {
            console.log('Invalid refresh token. Invalidating QuickBooks tokens.');
            // Clear the invalid tokens in the database
            await db.execute('DELETE FROM quickbooks_tokens WHERE user_id = ?', [req.user.id]);

            // Respond with a 401 status to indicate reauthorization is needed
            return res.status(401).json({ message: 'Please reconnect your QuickBooks account.' });
        }

        // If other errors occur, respond with a 500 error
        res.status(500).send('Failed to fetch expenses.');
    }
});


// Endpoint to check QuickBooks connection status
app.get('/quickbooks/status', async (req, res) => {
    try {
        const userId = req.user?.id || 1; // Replace with actual user ID logic
        const [[tokens]] = await db.execute(
            'SELECT access_token, realm_id FROM quickbooks_tokens WHERE user_id = ?',
            [userId]
        );

        if (tokens && tokens.access_token && tokens.realm_id) {
            res.json({ connected: true }); // User is connected to QuickBooks
        } else {
            res.json({ connected: false }); // Not connected
        }
    } catch (error) {
        console.error('Error checking QuickBooks connection status:', error.message);
        res.status(500).json({ connected: false, error: 'Failed to check QuickBooks connection status.' });
    }
});

