import React, { useState, useEffect, useCallback } from 'react';
import axios from 'axios';
import {
    Container,
    Typography,
    Table,
    TableBody,
    TableCell,
    TableContainer,
    TableHead,
    TableRow,
    Paper,
    Fab,
    IconButton,
    Dialog,
    DialogTitle,
    DialogContent,
    DialogActions,
    TextField,
    Button,
    MenuItem,
} from '@mui/material';
import { Add, Edit, Delete } from '@mui/icons-material';
import { useNavigate } from 'react-router-dom';

function Jobs() {
    const [jobs, setJobs] = useState([]);
    const [clients, setClients] = useState([]);
    const [open, setOpen] = useState(false);
    const [currentJob, setCurrentJob] = useState(null);
    const navigate = useNavigate();

    const fetchJobs = useCallback(async () => {
        try {
            const accessToken = localStorage.getItem('accessToken');
            if (!accessToken) throw new Error('Access token missing. Please log in again.');

            const response = await axios.get('http://localhost:5001/jobs', {
                headers: { Authorization: `Bearer ${accessToken}` },
            });
            setJobs(response.data);
        } catch (error) {
            if (error.response?.status === 401 || error.response?.status === 403) {
                console.error('Access denied. Refreshing token...');
                await refreshAuthToken();
                fetchJobs();
            } else {
                console.error('Error fetching jobs:', error.message);
            }
        }
    }, []);

    const fetchClients = useCallback(async () => {
        try {
            const accessToken = localStorage.getItem('accessToken');
            if (!accessToken) throw new Error('Access token missing. Please log in again.');

            const response = await axios.get('http://localhost:5001/clients', {
                headers: { Authorization: `Bearer ${accessToken}` },
            });
            setClients(response.data);
        } catch (error) {
            if (error.response?.status === 401 || error.response?.status === 403) {
                console.error('Access denied. Refreshing token...');
                await refreshAuthToken();
                fetchClients();
            } else {
                console.error('Error fetching clients:', error.message);
            }
        }
    }, []);

    const refreshAuthToken = useCallback(async () => {
        try {
            const refreshToken = localStorage.getItem('refreshToken');
            if (!refreshToken) throw new Error('Refresh token missing. Please log in again.');

            const response = await axios.post('http://localhost:5001/refresh-token', {}, {
                headers: { Authorization: `Bearer ${refreshToken}` },
            });

            const { accessToken } = response.data;
            localStorage.setItem('accessToken', accessToken);
            console.log('Token refreshed successfully.');
        } catch (error) {
            console.error('Error refreshing token:', error.message);
            localStorage.removeItem('accessToken');
            localStorage.removeItem('refreshToken');
            navigate('/login'); // Redirect to login page if token refresh fails
        }
    }, [navigate]);

    useEffect(() => {
        fetchJobs();
        fetchClients();

        const interval = setInterval(() => {
            refreshAuthToken();
        }, 50 * 60 * 1000); // Refresh token every 50 minutes
        return () => clearInterval(interval); // Cleanup on component unmount
    }, [fetchJobs, fetchClients, refreshAuthToken]);

    const handleRowClick = (job) => {
        navigate(`/jobs/${job.id}`);
    };

    const handleClickOpen = () => {
        setCurrentJob({ name: '', description: '', status: 'pending', client_id: '' });
        setOpen(true);
    };

    const handleClose = () => {
        setOpen(false);
        setCurrentJob(null);
    };

    const handleInputChange = (e) => {
        const { name, value } = e.target;
        setCurrentJob({ ...currentJob, [name]: value });
    };

    const handleSaveJob = async () => {
        try {
            const accessToken = localStorage.getItem('accessToken');
            if (!accessToken) throw new Error('Access token missing. Please log in again.');

            if (currentJob.id) {
                await axios.put(`http://localhost:5001/jobs/${currentJob.id}`, currentJob, {
                    headers: { Authorization: `Bearer ${accessToken}` },
                });
            } else {
                await axios.post('http://localhost:5001/jobs', currentJob, {
                    headers: { Authorization: `Bearer ${accessToken}` },
                });
            }
            fetchJobs();
            handleClose();
        } catch (error) {
            console.error('Error saving job:', error.message);
        }
    };

    return (
        <Container>
            <Typography variant="h4" gutterBottom>
                Jobs
            </Typography>
            <TableContainer component={Paper}>
                <Table>
                    <TableHead>
                        <TableRow>
                            <TableCell>Name</TableCell>
                            <TableCell>Client</TableCell>
                            <TableCell>Status</TableCell>
                            <TableCell>Actions</TableCell>
                        </TableRow>
                    </TableHead>
                    <TableBody>
                        {jobs.map((job) => (
                            <TableRow
                                key={job.id}
                                onClick={() => handleRowClick(job)}
                                style={{ cursor: 'pointer' }}
                            >
                                <TableCell>{job.name}</TableCell>
                                <TableCell>{job.client_name}</TableCell>
                                <TableCell>{job.status}</TableCell>
                                <TableCell>
                                    <IconButton
                                        onClick={(e) => {
                                            e.stopPropagation();
                                            setCurrentJob(job);
                                            setOpen(true);
                                        }}
                                    >
                                        <Edit />
                                    </IconButton>
                                    <IconButton
                                        onClick={(e) => {
                                            e.stopPropagation();
                                            // Add delete job logic here if needed
                                        }}
                                    >
                                        <Delete />
                                    </IconButton>
                                </TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
            </TableContainer>
            <Fab
                color="primary"
                aria-label="add"
                sx={{ position: 'fixed', bottom: 16, right: 16 }}
                onClick={handleClickOpen}
            >
                <Add />
            </Fab>
            <Dialog open={open} onClose={handleClose}>
                <DialogTitle>{currentJob?.id ? 'Edit Job' : 'Add Job'}</DialogTitle>
                <DialogContent>
                    <TextField
                        label="Name"
                        name="name"
                        fullWidth
                        margin="normal"
                        value={currentJob?.name || ''}
                        onChange={handleInputChange}
                    />
                    <TextField
                        label="Description"
                        name="description"
                        fullWidth
                        margin="normal"
                        multiline
                        rows={4}
                        value={currentJob?.description || ''}
                        onChange={handleInputChange}
                    />
                    <TextField
                        select
                        label="Client"
                        name="client_id"
                        fullWidth
                        margin="normal"
                        value={currentJob?.client_id || ''}
                        onChange={handleInputChange}
                    >
                        {clients.map((client) => (
                            <MenuItem key={client.id} value={client.id}>
                                {client.name}
                            </MenuItem>
                        ))}
                    </TextField>
                    <TextField
                        select
                        label="Status"
                        name="status"
                        fullWidth
                        margin="normal"
                        value={currentJob?.status || ''}
                        onChange={handleInputChange}
                    >
                        <MenuItem value="pending">Pending</MenuItem>
                        <MenuItem value="in-progress">In Progress</MenuItem>
                        <MenuItem value="completed">Completed</MenuItem>
                    </TextField>
                    <TextField
                        label="Start Date"
                        name="start_date"
                        type="date"
                        fullWidth
                        margin="normal"
                        InputLabelProps={{ shrink: true }}
                        value={currentJob?.start_date || ''}
                        onChange={handleInputChange}
                    />
                    <TextField
                        label="End Date"
                        name="end_date"
                        type="date"
                        fullWidth
                        margin="normal"
                        InputLabelProps={{ shrink: true }}
                        value={currentJob?.end_date || ''}
                        onChange={handleInputChange}
                    />
                </DialogContent>
                <DialogActions>
                    <Button onClick={handleClose} color="secondary">
                        Cancel
                    </Button>
                    <Button onClick={handleSaveJob} color="primary">
                        Save
                    </Button>
                </DialogActions>
            </Dialog>
        </Container>
    );
}

export default Jobs;
